/***************************************************************************
 *   Copyright (C) 2005-2006 Gao Xianchao                                  *
 *                 2007 Gao Xianchao gnap_an linux_lyb ahlongxp            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/*
 * Author:	gxc
 * Create data:	2005-10-11 20:09
 */
 
#include <signal.h>
#include "BTTask.h"
#include "TorrentFile.h"
#include "TrackerManager.h"
#include "EpollReactor.h"
#include "PeerManager.h"
#include "Storage.h"
#include "PeerAcceptor.h"
#include "UPnpNat.h"
#include "RateMeasure.h"
#include "utils.h"
#include "log.h"

void* TaskThreadFunc(void* param)
{
	sigset_t sigset;
	sigemptyset(&sigset);
	pthread_sigmask(SIG_SETMASK, &sigset, NULL);
	
	CBTTask* task = (CBTTask*)param;
	int ret = task->svc();
	pthread_exit(&ret);
	
	return NULL;
}

CBTTask::CBTTask()
: _torrentFile(NULL)
, _trackerManager(NULL)
{
}

CBTTask::~CBTTask()
{
}

void CBTTask::setTorrentFilePath(const char* path)
{
	_torrentFilePath = path;
}

 void CBTTask::setDestPath(const char* path)
 {
 	_destPath = path;
 	if(_destPath[_destPath.length()-1] != '/')
 	{
 		_destPath += "/";
 	}
 }
 
std::string CBTTask::getDestPath()
{
	return _destPath;
}

std::string CBTTask::getTaskName()
{
	return extractFileName(_torrentFilePath.c_str());
}

bool CBTTask::start()
{
	LOG_INFO("BTTask starting");
	
	_toExit = false;
	
	_downloadCount = 0;
	_uploadCount  = 0;
	_uploadSpeed = 0;
	_downloadSpeed = 0;
	 _lastUploadCount = 0;
	 _lastDownloadCount = 0;	
	_lastCheckSpeedTick = GetTickCount();
	_uploadSpeedList.clear();
	_downloadSpeedList.clear();	
	
	_peerLinkMax = 100;
	_uploadPeerLinkMax = 6;
	_cacheSize = 5*1024*1024;
	
	if(_torrentFile == NULL)
	{
		return false;
	}
	
	_socketReactor = new CEpollReactor();
	
	_rateMeasure = new CRateMeasure();
	
	_trackerManager = new CTrackerManager();
	_trackerManager->setBTTask(this);
	
	_peerManager = new CPeerManager();
	_peerManager->setBTTask(this);
	
	_storage = new CStorage();
	_storage->setBTTask(this);
	
	_acceptor = new CPeerAcceptor();
	_acceptor->setBTTask(this);
	
	_upnpNat = new CUPnpNat();
	_upnpNat->setReactor(_socketReactor);
	
	_storage->setBanedFileList(_bandFileList);
	if(!_storage->start())
	{
		_errorMessage = "Can not create/open download files";
		return false;
	}
	
	
	if(!_socketReactor->start())
	{
		_errorMessage = "Can not start SocketReactor";
		return false;		
	}
	
	_upnpNat->start();
	
	if(!_acceptor->start())
	{
		_errorMessage = "Can not start PeerAcceptor";
		return false;			
	}
	
	if(!_trackerManager->start())
	{
		_errorMessage = "Can not start TrackerManager";
		return false;			
	}
	
	if(!_peerManager->start())
	{
		_errorMessage = "Can not start PeerManager";
		return false;			
	}	
	
	_speedTimerID = _socketReactor->addTimer(this, 2000, false);
	
	pthread_create(&_threadHandle, NULL, TaskThreadFunc, this);
	

	LOG_INFO("BTTask started");
	
	return true;	
}

int CBTTask::svc()
{
	while(!_toExit)
	{
		_socketReactor->update();
		_rateMeasure->update();
	}
	
	return 0;
}

void CBTTask::stop()
{
	_toExit = true;
	if(_threadHandle != 0)
	{
		pthread_join(_threadHandle, NULL);
		_threadHandle = 0;
	}
		
	_acceptor->stop();
	_trackerManager->stop();
	_peerManager->stop();
	_upnpNat->stop();
	_socketReactor->stop();
	_storage->stop();	
	
	delete _trackerManager;
	_trackerManager = NULL;
	
	delete _peerManager;
	_peerManager = NULL;
	
	delete _trackerManager;
	_trackerManager = NULL;
	
	delete _socketReactor;
	_socketReactor = NULL;
	
	delete _rateMeasure;
	_rateMeasure = NULL;
	
	delete _storage;
	_storage = NULL;
	
	delete _torrentFile;
	_torrentFile = NULL;
	
	delete _upnpNat;
	_upnpNat = NULL;
}

std::string CBTTask::getErrorMessage()
{
	return _errorMessage;
}

bool CBTTask::loadTorrentFile()
{
	if(_torrentFile != NULL)
	{
		delete _torrentFile;
	}
	
	_torrentFile = new CTorrentFile();
	_torrentFile->setBTTask(this);
	
	if(!_torrentFile->load(filename_from_utf8(_torrentFilePath.c_str()).c_str()))
	{
		return false;
	}
	
	return true;
}

ITorrentFile* CBTTask::getTorrentFile()
{
	return _torrentFile;
}

void CBTTask::setBanedFileList(TBanedFileList bandFileList)
{
	_bandFileList = bandFileList;
}

ITrackerManager* CBTTask::getTrackerManager()
{
	return _trackerManager;
}

std::string CBTTask::getPeerID()
{
        if(_peerID.empty())
        {
	    char buf[21];
	    strcpy(buf, "-BM0002-");
	    *((int*)(buf+8)) = GetTickCount();
	    buf[20] = 0;
            _peerID = buf;
        }
	return _peerID;
}

ISocketReactor* CBTTask::getSocketReactor()
{
	return _socketReactor;
}

IPeerManager* CBTTask::getPeerManager()
{
	return _peerManager;
}

IStorage* CBTTask::getStorage()
{
	return _storage;
}

IPeerAcceptor*  CBTTask::getAcceptor()
{
	return _acceptor;
}

IUPnpNAT* CBTTask::getUPnpNat()
{
	return _upnpNat;
}

unsigned int CBTTask::getPeerLinkMax()
{
	return _peerLinkMax;
}

void CBTTask::setPeerLinkMax(unsigned int count)
{
	_peerLinkMax = count;
}

unsigned int CBTTask::getUploadPeerLinkMax()
{
	return _uploadPeerLinkMax;
}

void CBTTask::setUploadPeerLinkMax(unsigned int count)
{
	_uploadPeerLinkMax = count;
}

unsigned int CBTTask::getCacheSize()
{
	return _cacheSize;
}

void CBTTask::setCacheSize(unsigned int count)
{
	_cacheSize = count;
}

unsigned int CBTTask::getConnectingPeerLinkMax()
{
	return 20;
}

void CBTTask::incDownlaodCount(unsigned int count)
{
	_downloadCount += count;
}

void CBTTask::incUploadCount(unsigned int count)
{
	_uploadCount += count;
}

int64_t CBTTask::getDownlaodCount()
{
	return _downloadCount;
}

int64_t CBTTask::getUploadCount()
{
	return _uploadCount;
}

unsigned int CBTTask::checkUploadSpeed()
{
	unsigned int curSpeed =1000*(_uploadCount - _lastUploadCount)/ (GetTickCount()-_lastCheckSpeedTick);
	
	while(_uploadSpeedList.size()>=5)
	{
		_uploadSpeedList.pop_front();
	}
	_uploadSpeedList.push_back(curSpeed);
	
	unsigned int result = 0;
	TSpeedList::iterator iter = _uploadSpeedList.begin();
	for(; iter!= _uploadSpeedList.end(); ++iter)
	{
		result+=*iter;
	}
	
	return result / _uploadSpeedList.size();
}

unsigned int CBTTask::checkDownloadSpeed()
{
	unsigned int curSpeed =1000*(_downloadCount - _lastDownloadCount)/ (GetTickCount()-_lastCheckSpeedTick);
	
	while(_downloadSpeedList.size()>=5)
	{
		_downloadSpeedList.pop_front();
	}
	_downloadSpeedList.push_back(curSpeed);
	
	unsigned int result = 0;
	TSpeedList::iterator iter = _downloadSpeedList.begin();
	for(; iter!= _downloadSpeedList.end(); ++iter)
	{
		result+=*iter;
	}
	
	return result / _downloadSpeedList.size();
}

void CBTTask::onTimer(unsigned int id)
{
	if(id == _speedTimerID)
	{
		_uploadSpeed = checkUploadSpeed();
		_downloadSpeed = checkDownloadSpeed();		
		
		_lastCheckSpeedTick = GetTickCount();
		_lastUploadCount = _uploadCount;
		_lastDownloadCount = _downloadCount;
	}
}

unsigned int CBTTask::getUploadSpeed()
{
	return _uploadSpeed;
}

unsigned int CBTTask::getDownloadSpeed()
{
	return _downloadSpeed;
}

IRateMeasure* CBTTask::getRateMeasure()
{
	return _rateMeasure;
}

