#include "FFAudioDecoder.h"
#include "avm_output.h"
#include <stdlib.h>
#include <stdio.h>

AVM_BEGIN_NAMESPACE;

FFAudioDecoder::FFAudioDecoder(AVCodec* av, const CodecInfo& info, const WAVEFORMATEX* wf)
    :IAudioDecoder(info, wf), m_pAvCodec(av), m_pAvContext(0)
{
}

FFAudioDecoder::~FFAudioDecoder()
{
    if (m_pAvContext)
    {
	avcodec_close(m_pAvContext);
	free(m_pAvContext);
    }
}

int FFAudioDecoder::Convert(const void* in_data, size_t in_size,
			    void* out_data, size_t out_size,
			    size_t* size_read, size_t* size_written)
{
    if (!m_pAvContext)
    {
	if (!(m_pAvContext = avcodec_alloc_context2(m_pAvCodec->type)))
	    return -1;

	m_pAvContext->channels = m_pFormat->nChannels;
	if (m_pAvContext->channels > 2)
	    m_pAvContext->channels = 2;
	m_pAvContext->bit_rate = m_pFormat->nAvgBytesPerSec * 8;
	m_pAvContext->sample_rate = m_pFormat->nSamplesPerSec;
	m_pAvContext->block_align = m_pFormat->nBlockAlign;
	m_pAvContext->codec_id = m_pAvCodec->id;

	//printf("INITIALIZE ch:%d avg:%d sam:%d\n", m_pFormat->nChannels, m_pFormat->nAvgBytesPerSec, m_pFormat->nSamplesPerSec);
	if (m_pFormat->cbSize > 0)
	{
	    int skip = (m_pFormat->wFormatTag == WAVE_FORMAT_EXTENSIBLE) ?
		sizeof(FFMPEGWAVEFORMATEX) - sizeof(WAVEFORMATEX) : 0;

	    m_pAvContext->extradata = (uint8_t*)(m_pFormat + 1) + skip;
	    m_pAvContext->extradata_size = m_pFormat->cbSize - skip;
	}

	if (avcodec_open(m_pAvContext, m_pAvCodec) < 0)
	{
	    AVM_WRITE("FFAudioDecoder", "WARNING: can't open avcodec\n");
	    free(m_pAvContext);
	    m_pAvContext = 0;
            return -1;
	}
    }
    //printf("%0lx %0lx %0lx %0lx\n", ((long*)in_data)[0], ((long*)in_data)[1], ((long*)in_data)[2], ((long*)in_data)[3]);

    AVPacket avpkt;
    av_init_packet(&avpkt);
    avpkt.data = (uint8_t*)in_data;
    avpkt.size = (int)in_size;
    int framesz = AVCODEC_MAX_AUDIO_FRAME_SIZE;
    int hr = avcodec_decode_audio3(m_pAvContext, (int16_t*)out_data, &framesz, &avpkt);
    //printf("CONVERT  ins: %d  outs:%" PRIsz "  fs:%d   h:%d\n", avpkt.size, out_size, framesz, hr);
    if (size_read)
	*size_read = (hr < 0) ? 1 : hr;
    if (size_written)
	*size_written = (hr < 0) ? 0 : framesz;

    if (hr < 0)
    {
	avcodec_close(m_pAvContext);
	free(m_pAvContext);
        m_pAvContext = 0;
    }
    return (hr < 0 || in_size == 0) ? -1 : 0;
}

size_t FFAudioDecoder::GetMinSize() const
{
    size_t r;
    switch (m_Info.fourcc)
    {
    case WAVE_FORMAT_DVM:
	r = MIN_AC3_CHUNK_SIZE;
        break;
    case WAVE_FORMAT_MSAUDIO1:
	r = 16 * m_pFormat->nBlockAlign * (uint_t)m_uiBytesPerSec / m_pFormat->nAvgBytesPerSec;
        break;
    case WAVE_FORMAT_DVI_ADPCM:
	r = (!m_pFormat->nBlockAlign) ? 1024 : m_pFormat->nBlockAlign * m_pFormat->nChannels;
        break;
    default:
	r = 2;
    }
    //printf("minsize %d  0x%x\n", r, m_Info.fourcc);
    return (r < AVCODEC_MAX_AUDIO_FRAME_SIZE) ? AVCODEC_MAX_AUDIO_FRAME_SIZE : r;
}

size_t FFAudioDecoder::GetSrcSize(size_t dest_size) const
{
    switch (m_Info.fourcc)
    {
    case WAVE_FORMAT_MSAUDIO1:
    case WAVE_FORMAT_WMAUDIO2:
	return m_pFormat->nBlockAlign;
    case WAVE_FORMAT_DVI_ADPCM:
	if (!m_pFormat->nBlockAlign)
            return 1024;
    default:
	return IAudioDecoder::GetSrcSize(dest_size);
    }
}

AVM_END_NAMESPACE;
