// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CAST_STREAMING_SENDER_REPORT_PARSER_H_
#define CAST_STREAMING_SENDER_REPORT_PARSER_H_

#include "absl/types/optional.h"
#include "absl/types/span.h"
#include "cast/streaming/rtcp_common.h"
#include "cast/streaming/rtcp_session.h"
#include "cast/streaming/rtp_defines.h"
#include "cast/streaming/rtp_time.h"

namespace openscreen {
namespace cast {

// Parses RTCP packets from a Sender to extract Sender Reports. Ignores anything
// else, since that is all a Receiver would be interested in.
class SenderReportParser {
 public:
  // Returned by Parse(), to also separately expose the StatusReportId. The
  // report ID isn't included in the common RtcpSenderReport struct because it's
  // not an input to SenderReportBuilder (it is generated by the builder).
  struct SenderReportWithId : public RtcpSenderReport {
    SenderReportWithId();
    ~SenderReportWithId();

    StatusReportId report_id{};
  };

  explicit SenderReportParser(RtcpSession* session);
  ~SenderReportParser();

  // Parses the RTCP |packet|, and returns a parsed sender report if the packet
  // contained one. Returns nullopt if the data is corrupt or the packet did not
  // contain a sender report.
  absl::optional<SenderReportWithId> Parse(absl::Span<const uint8_t> packet);

 private:
  RtcpSession* const session_;

  // Tracks the recently-parsed RTP timestamps so that the truncated values can
  // be re-expanded into full-form.
  RtpTimeTicks last_parsed_rtp_timestamp_;
};

}  // namespace cast
}  // namespace openscreen

#endif  // CAST_STREAMING_SENDER_REPORT_PARSER_H_
